import {
  compact,
  pointInPolygon,
  polygonIntersectsPolyline,
  polygonsIntersect
} from "@tldraw/editor";
function kickoutOccludedShapes(editor, shapeIds) {
  const parentsToCheck = /* @__PURE__ */ new Set();
  for (const id of shapeIds) {
    const shape = editor.getShape(id);
    if (!shape) continue;
    if (editor.getShapeUtil(shape).onDragShapesOut) {
      parentsToCheck.add(shape);
    }
    const parent = editor.getShape(shape.parentId);
    if (!parent) continue;
    if (editor.getShapeUtil(parent).onDragShapesOut) {
      parentsToCheck.add(parent);
    }
  }
  const parentsWithKickedOutChildren = /* @__PURE__ */ new Map();
  for (const parent of parentsToCheck) {
    const occludedChildren = getOccludedChildren(editor, parent);
    if (occludedChildren.length) {
      parentsWithKickedOutChildren.set(parent, occludedChildren);
    }
  }
  for (const [parent, kickedOutChildrenIds] of parentsWithKickedOutChildren) {
    const shapeUtil = editor.getShapeUtil(parent);
    const kickedOutChildren = compact(kickedOutChildrenIds.map((id) => editor.getShape(id)));
    shapeUtil.onDragShapesOut?.(parent, kickedOutChildren);
  }
}
function getOccludedChildren(editor, parent) {
  const childIds = editor.getSortedChildIdsForParent(parent.id);
  if (childIds.length === 0) return [];
  const parentPageBounds = editor.getShapePageBounds(parent);
  if (!parentPageBounds) return [];
  let parentGeometry;
  let parentPageTransform;
  let parentPageCorners;
  const results = [];
  for (const childId of childIds) {
    const shapePageBounds = editor.getShapePageBounds(childId);
    if (!shapePageBounds) {
      continue;
    }
    if (!parentPageBounds.includes(shapePageBounds)) {
      results.push(childId);
      continue;
    }
    parentGeometry ??= editor.getShapeGeometry(parent);
    parentPageTransform ??= editor.getShapePageTransform(parent);
    parentPageCorners ??= parentPageTransform.applyToPoints(parentGeometry.vertices);
    const parentCornersInShapeSpace = editor.getShapePageTransform(childId).clone().invert().applyToPoints(parentPageCorners);
    const { vertices, isClosed } = editor.getShapeGeometry(childId);
    if (vertices.some((v) => pointInPolygon(v, parentCornersInShapeSpace))) {
      continue;
    }
    if (isClosed) {
      if (polygonsIntersect(parentCornersInShapeSpace, vertices)) {
        continue;
      }
    } else if (polygonIntersectsPolyline(parentCornersInShapeSpace, vertices)) {
      continue;
    }
    results.push(childId);
  }
  return results;
}
function startEditingShapeWithLabel(editor, shape, selectAll = false) {
  editor.select(shape);
  editor.setEditingShape(shape);
  editor.setCurrentTool("select.editing_shape", {
    target: "shape",
    shape
  });
  if (selectAll) {
    editor.emit("select-all-text", { shapeId: shape.id });
  }
  zoomToShapeIfOffscreen(editor);
}
const ZOOM_TO_SHAPE_PADDING = 16;
function zoomToShapeIfOffscreen(editor) {
  const selectionPageBounds = editor.getSelectionPageBounds();
  const viewportPageBounds = editor.getViewportPageBounds();
  if (selectionPageBounds && !viewportPageBounds.contains(selectionPageBounds)) {
    const eb = selectionPageBounds.clone().expandBy(ZOOM_TO_SHAPE_PADDING / editor.getZoomLevel()).expand(viewportPageBounds);
    const nextBounds = viewportPageBounds.clone().translate({
      x: (eb.center.x - viewportPageBounds.center.x) * 2,
      y: (eb.center.y - viewportPageBounds.center.y) * 2
    });
    editor.zoomToBounds(nextBounds, {
      animation: {
        duration: editor.options.animationMediumMs
      },
      inset: 0
    });
  }
}
export {
  getOccludedChildren,
  kickoutOccludedShapes,
  startEditingShapeWithLabel,
  zoomToShapeIfOffscreen
};
//# sourceMappingURL=selectHelpers.mjs.map
